<?php
header("Content-Type: application/json");
require "db.php";

$request_method = $_SERVER["REQUEST_METHOD"];

switch ($request_method) {
    case "GET":
        if (isset($_GET['news_id'])) {
            getNewsById($_GET['news_id']);
        } elseif (isset($_GET['category_id'])) {
            getCategoryById($_GET['category_id']);
        } else {
            getAllNews();
        }
        break;

    case "POST":
        if (isset($_GET['category'])) {
            createCategory();
        } else {
            createNews();
        }
        break;

    case "PUT":
        $input_data = json_decode(file_get_contents("php://input"), true);
        if (isset($input_data['news_id'])) {
            updateNews($input_data['news_id'], $input_data);
        } elseif (isset($input_data['category_id'])) {
            updateCategory($input_data['category_id'], $input_data);
        } else {
            http_response_code(400);
            echo json_encode(["error" => "ID is required"]);
        }
        break;

    case "DELETE":
        $input_data = json_decode(file_get_contents("php://input"), true);
        if (isset($input_data['news_id'])) {
            deleteNews($input_data['news_id']);
        } elseif (isset($input_data['category_id'])) {
            deleteCategory($input_data['category_id']);
        } else {
            http_response_code(400);
            echo json_encode(["error" => "ID is required"]);
        }
        break;

    default:
        http_response_code(405);
        echo json_encode(["error" => "Method Not Allowed"]);
        break;
}

// Fetch all news
function getAllNews() {
    global $conn;
    $stmt = $conn->prepare("
        SELECT n.*, c.category_name 
        FROM tbl_news n 
        LEFT JOIN tbl_category c ON n.category_id = c.category_id 
        ORDER BY n.news_date DESC
    ");
    $stmt->execute();
    $result = $stmt->get_result();
    echo json_encode($result->fetch_all(MYSQLI_ASSOC));
}

// Fetch single news by ID
function getNewsById($news_id) {
    global $conn;
    $stmt = $conn->prepare("SELECT * FROM tbl_news WHERE news_id = ?");
    $stmt->bind_param("i", $news_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $news = $result->fetch_assoc();
    
    if ($news) {
        echo json_encode($news);
    } else {
        http_response_code(404);
        echo json_encode(["error" => "News not found"]);
    }
}

// Create new news entry
function createNews() {
    global $conn;
    $input_data = json_decode(file_get_contents("php://input"), true);

    if (!isset($input_data['news_title']) || !isset($input_data['news_content'])) {
        http_response_code(400);
        echo json_encode(["error" => "Missing title or content"]);
        return;
    }

    $stmt = $conn->prepare("
        INSERT INTO tbl_news 
        (news_title, news_content, news_content_short, photo, banner, category_id, comment, meta_title, meta_keyword, meta_description, lang_id, stitle, title, img_alt, domain_id, encode_title) 
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
    ");
    $stmt->bind_param("sssssisississssi", 
        $input_data['news_title'], 
        $input_data['news_content'], 
        $input_data['news_content_short'], 
        $input_data['photo'], 
        $input_data['banner'], 
        $input_data['category_id'], 
        $input_data['comment'], 
        $input_data['meta_title'], 
        $input_data['meta_keyword'], 
        $input_data['meta_description'], 
        $input_data['lang_id'], 
        $input_data['stitle'], 
        $input_data['title'], 
        $input_data['img_alt'], 
        $input_data['domain_id'], 
        $input_data['encode_title']
    );

    if ($stmt->execute()) {
        echo json_encode(["message" => "News added successfully", "news_id" => $conn->insert_id]);
    } else {
        http_response_code(500);
        echo json_encode(["error" => "Failed to add news"]);
    }
}

// Update news by ID
function updateNews($news_id, $data) {
    global $conn;
    
    if (!isset($data['news_title']) || !isset($data['news_content'])) {
        http_response_code(400);
        echo json_encode(["error" => "Missing title or content"]);
        return;
    }

    $stmt = $conn->prepare("UPDATE tbl_news SET news_title = ?, news_content = ? WHERE news_id = ?");
    $stmt->bind_param("ssi", $data['news_title'], $data['news_content'], $news_id);

    if ($stmt->execute()) {
        echo json_encode(["message" => "News updated successfully"]);
    } else {
        http_response_code(500);
        echo json_encode(["error" => "Failed to update news"]);
    }
}

// Delete news by ID
function deleteNews($news_id) {
    global $conn;
    $stmt = $conn->prepare("DELETE FROM tbl_news WHERE news_id = ?");
    $stmt->bind_param("i", $news_id);

    if ($stmt->execute()) {
        echo json_encode(["message" => "News deleted successfully"]);
    } else {
        http_response_code(500);
        echo json_encode(["error" => "Failed to delete news"]);
    }
}

// Fetch single category by ID
function getCategoryById($category_id) {
    global $conn;
    $stmt = $conn->prepare("SELECT * FROM tbl_category WHERE category_id = ?");
    $stmt->bind_param("i", $category_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $category = $result->fetch_assoc();
    
    if ($category) {
        echo json_encode($category);
    } else {
        http_response_code(404);
        echo json_encode(["error" => "Category not found"]);
    }
}

// Create new category
function createCategory() {
    global $conn;
    $input_data = json_decode(file_get_contents("php://input"), true);

    if (!isset($input_data['category_name'])) {
        http_response_code(400);
        echo json_encode(["error" => "Missing category name"]);
        return;
    }

    $stmt = $conn->prepare("
        INSERT INTO tbl_category (category_name, category_banner, meta_title, meta_keyword, meta_description, lang_id) 
        VALUES (?, ?, ?, ?, ?, ?)
    ");
    $stmt->bind_param("sssssi", 
        $input_data['category_name'], 
        $input_data['category_banner'], 
        $input_data['meta_title'], 
        $input_data['meta_keyword'], 
        $input_data['meta_description'], 
        $input_data['lang_id']
    );

    if ($stmt->execute()) {
        echo json_encode(["message" => "Category added successfully", "category_id" => $conn->insert_id]);
    } else {
        http_response_code(500);
        echo json_encode(["error" => "Failed to add category"]);
    }
}

// Update category
function updateCategory($category_id, $data) {
    global $conn;
    
    $stmt = $conn->prepare("UPDATE tbl_category SET category_name = ?, category_banner = ?, meta_title = ?, meta_keyword = ?, meta_description = ?, lang_id = ? WHERE category_id = ?");
    $stmt->bind_param("ssssssi", $data['category_name'], $data['category_banner'], $data['meta_title'], $data['meta_keyword'], $data['meta_description'], $data['lang_id'], $category_id);

    if ($stmt->execute()) {
        echo json_encode(["message" => "Category updated successfully"]);
    } else {
        http_response_code(500);
        echo json_encode(["error" => "Failed to update category"]);
    }
}

// Delete category
function deleteCategory($category_id) {
    global $conn;
    $stmt = $conn->prepare("DELETE FROM tbl_category WHERE category_id = ?");
    $stmt->bind_param("i", $category_id);

    if ($stmt->execute()) {
        echo json_encode(["message" => "Category deleted successfully"]);
    } else {
        http_response_code(500);
        echo json_encode(["error" => "Failed to delete category"]);
    }
}
?>
